/*
  pins_arduino.h - Pin definition functions for Arduino
  Part of Arduino - http://www.arduino.cc/

  Copyright (c) 2007 David A. Mellis

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General
  Public License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place, Suite 330,
  Boston, MA  02111-1307  USA

  $Id: wiring.h 249 2007-02-03 16:52:51Z mellis $

  modified for Uno1284P 30/jan/2013 by I.Kato @ BS21 Lab
*/

#ifndef Pins_Arduino_h
#define Pins_Arduino_h

#include <avr/pgmspace.h>

#define __AVR_ATmega1284P_UNO1284P__

#define NUM_DIGITAL_PINS            32
#define NUM_ANALOG_INPUTS           8
#define analogInputToDigitalPin(p)  ((p < 8) ? ((p) + 24) : -1)

#define digitalPinHasPWM(p)         ((p) == 4 || (p) == 5 || (p) == 6 || (p) == 8 || (p) == 9 || (p) == 10 || (p) == 12 || (p) == 13)

const static uint8_t SS   = 10;
const static uint8_t MOSI = 11;
const static uint8_t MISO = 12;
const static uint8_t SCK  = 13;

const static uint8_t SDA = 17;
const static uint8_t SCL = 16;
const static uint8_t LED_BUILTIN = 13;

const static uint8_t A0 = 24;
const static uint8_t A1 = 25;
const static uint8_t A2 = 26;
const static uint8_t A3 = 27;
const static uint8_t A4 = 28;
const static uint8_t A5 = 29;
const static uint8_t A6 = 30;
const static uint8_t A7 = 31;


extern const uint8_t PROGMEM digital_pin_to_PCMSK_PGM[];
extern const uint8_t PROGMEM digital_pin_to_PCI_bit_PGM[];

#define digitalPinToPCICR(p)    (((p) >= 0 && (p) <= 31) ? (&PCICR) : ((uint8_t *)0))
#define digitalPinToPCICRbit(p) (*pgm_read_byte(digital_pin_to_PCI_bit_PGM + (p)) >> 4)

#define digitalPinToPCMSK(p)    (((p) >= 0 && (p) <= 31) ? ((uint8_t *)pgm_read_byte(digital_pin_to_PCMSK_PGM + (p))) : ((uint8_t *)0))
#define digitalPinToPCMSKbit(p) (*pgm_read_byte(digital_pin_to_PCI_bit_PGM + (p)) & 0x07)


#ifdef ARDUINO_MAIN
// wiring_digital.c をコンパイルする時のみ通る。

// On the Arduino board, digital pins are also used
// for the analog output (software PWM).  Analog input
// pins are a separate set.

// ATMEL ATMEGA1284P / Uno1284P
//
//                             +---\/---+
//         T0 (D14)      PB0  1|        |40  PA0 (A0 / D24) ADC0
//         T1 (D15)      PB1  2|        |39  PA1 (A1 / D25) ADC1
//  INT2/AIN0 (D7)       PB2  3|        |38  PA2 (A2 / D26) ADC2
//  PWM6/AIN1 (D6)       PB3  4|        |37  PA3 (A3 / D27) ADC3
//   PWM5/#SS (D5)       PB4  5|        |36  PA4 (A6 / D30) ADC4
//       MOSI (D11<=>D8) PB5  6|        |35  PA5 (A7 / D31) ADC5
// PWM12/MISO (D12)      PB6  7|        |34  PA6 (A4 / D28) ADC6
//  PWM13/SCK (D13)      PB7  8|        |33  PA7 (A5 / D29) ADC7
//                      #RST  9|        |32  AREF
//                       VCC 10|        |31  GND
//                       GND 11|        |30  AVCC
//                     XTAL2 12|        |29  PC7 (D23) TOSC2
//                     XTAL1 13|        |28  PC6 (D22) TOSC1
//     T3/RX0 (D0)       PD0 14|        |27  PC5 (D21) TDI
//        TX0 (D1)       PD1 15|        |26  PC4 (D20) TDO
//   INT0/RX1 (D2)       PD2 16|        |25  PC3 (D19) TMS
//   INT1/TX1 (D3<=>D4)  PD3 17|        |24  PC2 (D18) TCK
//      PWM10 (D10)      PD4 18|        |23  PC1 (D17) SDA
//       PWM9 (D9)       PD5 19|        |22  PC0 (D16) SCL
//       PWM8 (D8<=>D11) PD6 20|        |21  PD7 (D4<=>D3) PWM4
//                             +--------+
//


// these arrays map port names (e.g. port B) to the
// appropriate addresses for various functions (e.g. reading
// and writing)
const uint16_t PROGMEM port_to_mode_PGM[] = {
   NOT_A_PORT,
   (uint16_t) &DDRA,
   (uint16_t) &DDRB,
   (uint16_t) &DDRC,
   (uint16_t) &DDRD,
};

const uint16_t PROGMEM port_to_output_PGM[] = {
   NOT_A_PORT,
   (uint16_t) &PORTA,
   (uint16_t) &PORTB,
   (uint16_t) &PORTC,
   (uint16_t) &PORTD,
};

const uint16_t PROGMEM port_to_input_PGM[] = {
   NOT_A_PORT,
   (uint16_t) &PINA,
   (uint16_t) &PINB,
   (uint16_t) &PINC,
   (uint16_t) &PIND,
};

const uint8_t PROGMEM digital_pin_to_port_PGM[] = {
   PD, /* 0 */
   PD,
   PD,
   PD,
   PD,

   PB, /* 5 */
   PB,
   PB,

   PD, /* 8 */
   PD,
   PD,

   PB, /* 11 */
   PB,
   PB,
   PB,
   PB,

   PC, /* 16 */
   PC,
   PC,
   PC,
   PC,
   PC,
   PC,
   PC,

   PA, /* 24 */
   PA,
   PA,
   PA,
   PA,
   PA,
   PA,
   PA,
};

const uint8_t PROGMEM digital_pin_to_bit_mask_PGM[] = {
   _BV(0), /* 0, port D */
   _BV(1),
   _BV(2),
   _BV(3),
   _BV(7),

   _BV(4), /* 5, port B */
   _BV(3),
   _BV(2),

   _BV(6), /* 8, port D */
   _BV(5),
   _BV(4),

   _BV(5), /* 11, port B */
   _BV(6),
   _BV(7),
   _BV(0),
   _BV(1),

   _BV(0), /* 16, port C */
   _BV(1),
   _BV(2),
   _BV(3),
   _BV(4),
   _BV(5),
   _BV(6),
   _BV(7),

   _BV(0), /* 24, port A */
   _BV(1),
   _BV(2),
   _BV(3),
   _BV(6),
   _BV(7),
   _BV(4),
   _BV(5),
};

const uint8_t PROGMEM digital_pin_to_timer_PGM[] = {
   NOT_ON_TIMER,   /* 0  - PD0 */
   NOT_ON_TIMER,   /* 1  - PD1 */
   NOT_ON_TIMER,   /* 2  - PD2 */
   NOT_ON_TIMER,   /* 3  - PD3 */
   TIMER2A,        /* 4  - PD7 */
   TIMER0B,        /* 5  - PB4 */
   TIMER0A,        /* 6  - PB3 */
   NOT_ON_TIMER,   /* 7  - PB2 */
   TIMER2B,        /* 8  - PD6 */
   TIMER1A,        /* 9  - PD5 */
   TIMER1B,        /* 10 - PD4 */
   NOT_ON_TIMER,   /* 11 - PB5 */
   TIMER3A,        /* 12 - PB6 */
   TIMER3B,        /* 13 - PB7 */
   NOT_ON_TIMER,   /* 14 - PB0 */
   NOT_ON_TIMER,   /* 15 - PB1 */
   NOT_ON_TIMER,   /* 16 - PC0 */
   NOT_ON_TIMER,   /* 17 - PC1 */
   NOT_ON_TIMER,   /* 18 - PC2 */
   NOT_ON_TIMER,   /* 19 - PC3 */
   NOT_ON_TIMER,   /* 20 - PC4 */
   NOT_ON_TIMER,   /* 21 - PC5 */
   NOT_ON_TIMER,   /* 22 - PC6 */
   NOT_ON_TIMER,   /* 23 - PC7 */
   NOT_ON_TIMER,   /* 24 - PA0 */
   NOT_ON_TIMER,   /* 25 - PA1 */
   NOT_ON_TIMER,   /* 26 - PA2 */
   NOT_ON_TIMER,   /* 27 - PA3 */
   NOT_ON_TIMER,   /* 28 - PA6 */
   NOT_ON_TIMER,   /* 29 - PA7 */
   NOT_ON_TIMER,   /* 30 - PA4 */
   NOT_ON_TIMER,   /* 31 - PA5 */
};

const uint8_t PROGMEM digital_pin_to_PCMSK_PGM[] = {
   (uint8_t)(uint16_t) &PCMSK3,   /* 0  - PD0 */
   (uint8_t)(uint16_t) &PCMSK3,   /* 1  - PD1 */
   (uint8_t)(uint16_t) &PCMSK3,   /* 2  - PD2 */
   (uint8_t)(uint16_t) &PCMSK3,   /* 3  - PD3 */
   (uint8_t)(uint16_t) &PCMSK3,   /* 4  - PD7 */
   (uint8_t)(uint16_t) &PCMSK1,   /* 5  - PB4 */
   (uint8_t)(uint16_t) &PCMSK1,   /* 6  - PB3 */
   (uint8_t)(uint16_t) &PCMSK1,   /* 7  - PB2 */
   (uint8_t)(uint16_t) &PCMSK3,   /* 8  - PD6 */
   (uint8_t)(uint16_t) &PCMSK3,   /* 9  - PD5 */
   (uint8_t)(uint16_t) &PCMSK3,   /* 10 - PD4 */
   (uint8_t)(uint16_t) &PCMSK1,   /* 11 - PB5 */
   (uint8_t)(uint16_t) &PCMSK1,   /* 12 - PB6 */
   (uint8_t)(uint16_t) &PCMSK1,   /* 13 - PB7 */
   (uint8_t)(uint16_t) &PCMSK1,   /* 14 - PB0 */
   (uint8_t)(uint16_t) &PCMSK1,   /* 15 - PB1 */
   (uint8_t)(uint16_t) &PCMSK2,   /* 16 - PC0 */
   (uint8_t)(uint16_t) &PCMSK2,   /* 17 - PC1 */
   (uint8_t)(uint16_t) &PCMSK2,   /* 18 - PC2 */
   (uint8_t)(uint16_t) &PCMSK2,   /* 19 - PC3 */
   (uint8_t)(uint16_t) &PCMSK2,   /* 20 - PC4 */
   (uint8_t)(uint16_t) &PCMSK2,   /* 21 - PC5 */
   (uint8_t)(uint16_t) &PCMSK2,   /* 22 - PC6 */
   (uint8_t)(uint16_t) &PCMSK2,   /* 23 - PC7 */
   (uint8_t)(uint16_t) &PCMSK0,   /* 24 - PA0 */
   (uint8_t)(uint16_t) &PCMSK0,   /* 25 - PA1 */
   (uint8_t)(uint16_t) &PCMSK0,   /* 26 - PA2 */
   (uint8_t)(uint16_t) &PCMSK0,   /* 27 - PA3 */
   (uint8_t)(uint16_t) &PCMSK0,   /* 28 - PA6 */
   (uint8_t)(uint16_t) &PCMSK0,   /* 29 - PA7 */
   (uint8_t)(uint16_t) &PCMSK0,   /* 30 - PA4 */
   (uint8_t)(uint16_t) &PCMSK0,   /* 31 - PA5 */
};

const uint8_t PROGMEM digital_pin_to_PCI_bit_PGM[] = {
   (PCIE3 << 4) + PCINT24,   /* 0  - PD0 */
   (PCIE3 << 4) + PCINT25,   /* 1  - PD1 */
   (PCIE3 << 4) + PCINT26,   /* 2  - PD2 */
   (PCIE3 << 4) + PCINT27,   /* 3  - PD3 */
   (PCIE3 << 4) + PCINT31,   /* 4  - PD7 */
   (PCIE1 << 4) + PCINT12,   /* 5  - PB4 */
   (PCIE1 << 4) + PCINT11,   /* 6  - PB3 */
   (PCIE1 << 4) + PCINT10,   /* 7  - PB2 */
   (PCIE3 << 4) + PCINT30,   /* 8  - PD6 */
   (PCIE3 << 4) + PCINT29,   /* 9  - PD5 */
   (PCIE3 << 4) + PCINT28,   /* 10 - PD4 */
   (PCIE1 << 4) + PCINT13,   /* 11 - PB5 */
   (PCIE1 << 4) + PCINT14,   /* 12 - PB6 */
   (PCIE1 << 4) + PCINT15,   /* 13 - PB7 */
   (PCIE1 << 4) + PCINT8,    /* 14 - PB0 */
   (PCIE1 << 4) + PCINT9,    /* 15 - PB1 */
   (PCIE2 << 4) + PCINT16,   /* 16 - PC0 */
   (PCIE2 << 4) + PCINT17,   /* 17 - PC1 */
   (PCIE2 << 4) + PCINT18,   /* 18 - PC2 */
   (PCIE2 << 4) + PCINT19,   /* 19 - PC3 */
   (PCIE2 << 4) + PCINT20,   /* 20 - PC4 */
   (PCIE2 << 4) + PCINT21,   /* 21 - PC5 */
   (PCIE2 << 4) + PCINT22,   /* 22 - PC6 */
   (PCIE2 << 4) + PCINT23,   /* 23 - PC7 */
   (PCIE0 << 4) + PCINT0,    /* 24 - PA0 */
   (PCIE0 << 4) + PCINT1,    /* 25 - PA1 */
   (PCIE0 << 4) + PCINT2,    /* 26 - PA2 */
   (PCIE0 << 4) + PCINT3,    /* 27 - PA3 */
   (PCIE0 << 4) + PCINT6,    /* 28 - PA6 */
   (PCIE0 << 4) + PCINT7,    /* 29 - PA7 */
   (PCIE0 << 4) + PCINT4,    /* 30 - PA4 */
   (PCIE0 << 4) + PCINT5,    /* 31 - PA5 */
};


#endif /* ARDUINO_MAIN */


// Service Functions for High Program Memory Data that allocated over 64kB. 128kB AVR only.
// ja: 64kB 以上に配置されている High Program Memory データを使用する為のサービス関数群。128kB AVR 専用。
//
// Program Memory Map (Byte addressing)
//   0x00000: .vectors section
//   0x0uuuu: .progmem.data(=PROGMEM) section
//   0x0vvvv: .init0-9 sections
//   0x0wwww: .text section
//   0x1xxxx: .fini9-1 sections
//   0x1yyyy: .fini0(=HIGH_PROGMEM) section (This section must be located over 64kB.)
//   0x1zzzz: space
//   0x1FC00: OptiBootloader for Uno1284P
//   0x1FFFF: memory end


// Allocates Program Data on HIGHER address than the ".text" section. You need to allocate the ".text" section stepped over 64kB border.
// ja: .text セクションよりも高位アドレスに配置されるようにする。ただし、.text セクション自体が 64kB 境界をまたぐ様にする必要がある。
// Notes: DO NOT use ***_P functions of avr/pgmspace.h for High Program Memory Data. Only use hi_pgm_read_*** functions.
// ja: 注意点：avr/pgmspace.h の ***_P 関数群ではアクセスできない。後述の hi_pgm_read_*** 関数群でのアクセスのみ可能。
#define HIGH_PROGMEM __attribute__((section(".fini0")))

// RAMPZ is fixed to '1'. Because RAMPZ has not effected to LPM instruction.
// ja: RAMPZ は LPM 命令に影響しないので 1 に固定しておく。128kB AVR では 0 か 1 以外にはならず、さらに 0 の時は LPM 命令で事足りる為。ただし、SPM 命令を使う時は要注意。
#define UseHighProgMem()  (RAMPZ = 1)


// You need to call UseHighProgMem function once time, before these functions using.
// ja: 以下の関数は、先に一回 UseHighProgMem 関数を呼び出して RAMPZ を 1 にしておかないと、意図した通りに動作しない

#define __ELPM_byte_enhanced_simple__(addr) \
(__extension__({                            \
    uint16_t __addr16 = (uint16_t)(addr);   \
    uint8_t __result;                       \
    __asm__                                 \
    (                                       \
        "movw r30, %1" "\n\t"               \
        "elpm %0, Z+" "\n\t"                \
        : "=r" (__result)                   \
        : "r" (__addr16)                    \
        : "r30", "r31"                      \
    );                                      \
    __result;                               \
}))

#define __ELPM_word_enhanced_simple__(addr) \
(__extension__({                            \
    uint16_t __addr16 = (uint16_t)(addr);   \
    uint16_t __result;                      \
    __asm__                                 \
    (                                       \
        "movw r30, %1"  "\n\t"              \
        "elpm %A0, Z+"  "\n\t"              \
        "elpm %B0, Z"   "\n\t"              \
        : "=r" (__result)                   \
        : "r" (__addr16)                    \
        : "r30", "r31"                      \
    );                                      \
    __result;                               \
}))


#define __ELPM_dword_enhanced_simple__(addr) \
(__extension__({                             \
    uint16_t __addr16 = (uint16_t)(addr);    \
    uint32_t __result;                       \
    __asm__                                  \
    (                                        \
        "movw r30, %1"  "\n\t"               \
        "elpm %A0, Z+"  "\n\t"               \
        "elpm %B0, Z+"  "\n\t"               \
        "elpm %C0, Z+"  "\n\t"               \
        "elpm %D0, Z"   "\n\t"               \
        : "=r" (__result)                    \
        : "r" (__addr16)                     \
        : "r30", "r31"                       \
    );                                       \
    __result;                                \
}))


#define __ELPM_float_enhanced_simple__(addr) \
(__extension__({                             \
    uint16_t __addr16 = (uint16_t)(addr);    \
    float __result;                          \
    __asm__                                  \
    (                                        \
        "movw r30, %1"  "\n\t"               \
        "elpm %A0, Z+"  "\n\t"               \
        "elpm %B0, Z+"  "\n\t"               \
        "elpm %C0, Z+"  "\n\t"               \
        "elpm %D0, Z"   "\n\t"               \
        : "=r" (__result)                    \
        : "r" (__addr16)                     \
        : "r30", "r31"                       \
    );                                       \
    __result;                                \
}))

#define __ELPM_byte_simple(addr)   __ELPM_byte_enhanced_simple__(addr)
#define __ELPM_word_simple(addr)   __ELPM_word_enhanced_simple__(addr)
#define __ELPM_dword_simple(addr)  __ELPM_dword_enhanced_simple__(addr)
#define __ELPM_float_simple(addr)  __ELPM_float_enhanced_simple__(addr)

#define pgm_read_byte_hi(address_short)  __ELPM_byte_simple(address_short)
#define pgm_read_word_hi(address_short)  __ELPM_word_simple(address_short)
#define pgm_read_dword_hi(address_short) __ELPM_dword_simple(address_short)
#define pgm_read_float_hi(address_short) __ELPM_float_simple(address_short)


#ifdef __cplusplus
extern "C" {
#endif

// copies from High Program Memory Data to SRAM memory.
// ja: High Program Memory データを SRAM 上のメモリにコピーする
extern void * memcpy_PH (void * dest, const void * src, size_t len);

#ifdef __cplusplus
}
#endif


#ifdef ARDUINO_MAIN
// wiring_digital.c をコンパイルする時のみ通る。

#ifdef __cplusplus
extern "C" {
#endif

// Allocates the forever loop function in .fini1 section, because High Program Memory Data is allocated in .fini0 section.
// ja: High Program Memory データが .fini0 セクションに配置されるので、exit 関数が暴走しないように .fini1 セクションに無限ループする関数を配置しておく。
void stop_program_in_fini1_sec(void) __attribute__ ((naked)) __attribute__ ((section (".fini1")));

void stop_program_in_fini1_sec(void)
{
  cli(); // interrupt disable
  while (1) ;  // forever loop
}


// copies from High Program Memory Data to SRAM memory.
// ja: High Program Memory データを SRAM 上のメモリにコピーする
void * memcpy_PH (void * dest, const void * src, size_t len)
{
  __asm__ __volatile__ (
    "   movw r30, %[src]\n"
    "   movw r26, %[dest]\n"
    "   rjmp 2f\n"
    "1: elpm r0, Z+\n"
    "   st   X+, r0\n"
    "2: subi %A[len], 1\n"
    "   sbci %B[len], 0\n"
    "   brcc 1b\n"
    :
    : [dest] "r" (dest),
      [src]  "r" (src),
      [len]  "r" (len)
    : "r0", "r26", "r27", "r30", "r31"
  );
  return dest;
}

#ifdef __cplusplus
}
#endif


#endif /* ARDUINO_MAIN */


#endif /* Pins_Arduino_h */
